package dev.felnull.otyacraftengine.client.util;

import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.blaze3d.vertex.*;
import dev.felnull.otyacraftengine.client.renderer.OERenderTypes;
import dev.felnull.otyacraftengine.explatform.client.OEClientExpectPlatform;
import org.jetbrains.annotations.NotNull;
import org.joml.Matrix4f;

import java.util.Objects;
import java.util.UUID;
import java.util.function.Consumer;
import net.minecraft.class_1007;
import net.minecraft.class_1087;
import net.minecraft.class_1306;
import net.minecraft.class_1799;
import net.minecraft.class_2350;
import net.minecraft.class_2561;
import net.minecraft.class_2680;
import net.minecraft.class_2741;
import net.minecraft.class_286;
import net.minecraft.class_287;
import net.minecraft.class_289;
import net.minecraft.class_290;
import net.minecraft.class_293;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3532;
import net.minecraft.class_4587;
import net.minecraft.class_4588;
import net.minecraft.class_4597;
import net.minecraft.class_757;
import net.minecraft.class_778;
import net.minecraft.class_7833;
import net.minecraft.class_809;

/**
 * 描画関係のユーティリティ
 *
 * @author MORIMORI0317
 */
public final class OERenderUtils {
    private static final class_310 mc = class_310.method_1551();
    public static ThreadLocal<Boolean> SKIP_TRANSANDROT_MODELPART = ThreadLocal.withInitial(() -> false);
    public static final float MIN_BREADTH = 1.0E-3F;

    /**
     * PoseStackを16分の１単位で移動する
     *
     * @param poseStack PoseStack
     * @param x         X
     * @param y         Y
     * @param z         Z
     */
    public static void poseTrans16(@NotNull class_4587 poseStack, double x, double y, double z) {
        float pix = 1f / 16f;
        poseStack.method_22904(pix * x, pix * y, pix * z);
    }

    /**
     * PoseStackをすべてのスケールを設定する
     *
     * @param poseStack PoseStack
     * @param scale     すべてのスケール
     */
    public static void poseScaleAll(@NotNull class_4587 poseStack, float scale) {
        poseStack.method_22905(scale, scale, scale);
    }

    /**
     * PoseStackの角度をそれぞれ設定する
     *
     * @param poseStack PoseStack
     * @param x         X角度
     * @param y         Y角度
     * @param z         Z角度
     */
    public static void poseRotateAll(@NotNull class_4587 poseStack, float x, float y, float z) {
        poseRotateX(poseStack, x);
        poseRotateY(poseStack, y);
        poseRotateZ(poseStack, z);
    }

    /**
     * PoseStackのX角度を設定する
     *
     * @param poseStack PoseStack
     * @param angle     角度
     */
    public static void poseRotateX(@NotNull class_4587 poseStack, float angle) {
        poseStack.method_22907(class_7833.field_40714.rotationDegrees(angle));
    }

    /**
     * PoseStackのY角度を設定する
     *
     * @param poseStack PoseStack
     * @param angle     角度
     */
    public static void poseRotateY(@NotNull class_4587 poseStack, float angle) {
        poseStack.method_22907(class_7833.field_40716.rotationDegrees(angle));
    }

    /**
     * PoseStackのZ角度を設定する
     *
     * @param poseStack PoseStack
     * @param angle     角度
     */
    public static void poseRotateZ(@NotNull class_4587 poseStack, float angle) {
        poseStack.method_22907(class_7833.field_40718.rotationDegrees(angle));
    }

    /**
     * PoseStackの方向のブロックステートに設定する
     *
     * @param poseStack PoseStack
     * @param state     角度
     * @param roted     回転ずれ
     */
    public static void poseRotateHorizontalState(@NotNull class_4587 poseStack, @NotNull class_2680 state, int roted) {
        class_2350 direction = state.method_11654(class_2741.field_12481);
        poseRotateDirection(poseStack, direction, roted);
    }

    /**
     * PoseStackをDirectionの方向にする
     *
     * @param poseStack PoseStack
     * @param direction 方向
     * @param roted     回転ずれ
     */
    public static void poseRotateDirection(@NotNull class_4587 poseStack, @NotNull class_2350 direction, int roted) {
        for (int i = 0; i < roted; i++) {
            direction = direction.method_10170();
        }
        if (direction == class_2350.field_11039) {
            poseRotateY(poseStack, 180);
            poseStack.method_46416(-1f, 0f, -1f);
        } else if (direction == class_2350.field_11043) {
            poseRotateY(poseStack, 90);
            poseStack.method_46416(-1f, 0f, 0f);
        } else if (direction == class_2350.field_11035) {
            poseRotateY(poseStack, 270);
            poseStack.method_46416(0f, 0f, -1f);
        }
    }

    /**
     * 中心でposeを変更する
     *
     * @param poseStack         PoseStack
     * @param centerX           中心X
     * @param centerY           中心Y
     * @param centerZ           中心Z
     * @param poseStackConsumer 中心での処理
     */
    public static void poseCenterConsumer(@NotNull class_4587 poseStack, float centerX, float centerY, float centerZ, @NotNull Consumer<class_4587> poseStackConsumer) {
        poseStack.method_46416(centerX, centerY, centerZ);
        poseStackConsumer.accept(poseStack);
        poseStack.method_46416(-centerX, -centerY, -centerZ);
    }

    /**
     * GUI上でテクスチャを描画する
     *
     * @param location      テクスチャ
     * @param poseStack     PoseStack
     * @param x             X
     * @param y             Y
     * @param u0            テクスチャの開始地点X
     * @param v0            テクスチャの開始地点Y
     * @param ru1           相対テクスチャの終了地点X
     * @param rv1           相対テクスチャの終了地点Y
     * @param textureWidth  テクスチャの横サイズ
     * @param textureHeight テクスチャの縦サイズ
     */
    public static void drawTexture(@NotNull class_2960 location, @NotNull class_4587 poseStack, float x, float y, float u0, float v0, float ru1, float rv1, float textureWidth, float textureHeight) {
        setPreDraw(location);
        blitFloat(poseStack, x, y, u0, v0, ru1, rv1, textureWidth, textureHeight);
    }

    /**
     * GUI上でテクスチャを描画する
     * テクスチャサイズは256x256
     *
     * @param location  テクスチャ
     * @param poseStack PoseStack
     * @param x         X
     * @param y         Y
     * @param ru0       相対テクスチャの開始地点X
     * @param rv0       相対テクスチャの開始地点Y
     * @param ru1       テクスチャの終了地点X
     * @param rv1       テクスチャの終了地点Y
     */
    public static void drawTexture(@NotNull class_2960 location, @NotNull class_4587 poseStack, float x, float y, float ru0, float rv0, float ru1, float rv1) {
        drawTexture(location, poseStack, x, y, ru0, rv0, ru1, rv1, 256, 256);
    }

    /**
     * テクスチャ描画前に呼び出し
     *
     * @param location テクスチャ
     */
    public static void setPreDraw(@NotNull class_2960 location) {
        RenderSystem.setShader(class_757::method_34542);
        RenderSystem.setShaderTexture(0, location);
        RenderSystem.setShaderColor(1f, 1f, 1f, 1f);
    }

    /**
     * 半透明で描画のための呼び出し
     *
     * @param location テクスチャ
     * @param draw     描画処理
     */
    public static void setAndDrawAlpha(@NotNull class_2960 location, Runnable draw) {
        setPreDraw(location);
        RenderSystem.enableBlend();
        RenderSystem.defaultBlendFunc();
        draw.run();
        RenderSystem.disableBlend();
    }

    /**
     * GUI上で半透明テクスチャを描画する
     *
     * @param location      テクスチャ
     * @param poseStack     PoseStack
     * @param x             X
     * @param y             Y
     * @param u0            テクスチャの開始地点X
     * @param v0            テクスチャの開始地点Y
     * @param ru1           相対テクスチャの終了地点X
     * @param rv1           相対テクスチャの終了地点Y
     * @param textureWidth  テクスチャの横サイズ
     * @param textureHeight テクスチャの縦サイズ
     */
    public static void drawTextureAlpha(@NotNull class_2960 location, @NotNull class_4587 poseStack, float x, float y, float u0, float v0, float ru1, float rv1, float textureWidth, float textureHeight) {
        setAndDrawAlpha(location, () -> blitFloat(poseStack, x, y, u0, v0, ru1, rv1, textureWidth, textureHeight));
    }

    /**
     * GUI上で半透明テクスチャを描画する
     * テクスチャサイズは256x256
     *
     * @param location  テクスチャ
     * @param poseStack PoseStack
     * @param x         X
     * @param y         Y
     * @param u0        相対テクスチャの開始地点X
     * @param v0        相対テクスチャの開始地点Y
     * @param ru1       テクスチャの終了地点X
     * @param rv1       テクスチャの終了地点Y
     */
    public static void drawTextureAlpha(@NotNull class_2960 location, @NotNull class_4587 poseStack, float x, float y, float u0, float v0, float ru1, float rv1) {
        drawTextureAlpha(location, poseStack, x, y, u0, v0, ru1, rv1, 256, 256);
    }

    /**
     * 指定済みテクスチャとシェーダーで描画
     *
     * @param poseStack     PoseStack
     * @param x             X
     * @param y             Y
     * @param u0            テクスチャの開始地点X
     * @param v0            テクスチャの開始地点Y
     * @param rvu1          相対テクスチャの終了地点X
     * @param rvv1          相対テクスチャの終了地点Y
     * @param textureWidth  テクスチャの横サイズ
     * @param textureHeight テクスチャの縦サイズ
     */
    public static void blitFloat(@NotNull class_4587 poseStack, float x, float y, float u0, float v0, float rvu1, float rvv1, float textureWidth, float textureHeight) {
        Matrix4f matrix4f = poseStack.method_23760().method_23761();
        float ry = x + rvu1;
        float rh = y + rvv1;
        float ru0 = u0 / textureWidth;
        float ru1 = (u0 + rvu1) / textureWidth;
        float rv0 = v0 / textureHeight;
        float rv1 = (v0 + rvv1) / textureHeight;

        RenderSystem.setShader(class_757::method_34542);
        class_287 bufferBuilder = class_289.method_1348().method_1349();
        bufferBuilder.method_1328(class_293.class_5596.field_27382, class_290.field_1585);
        bufferBuilder.method_22918(matrix4f, x, rh, 0).method_22913(ru0, rv1).method_1344();
        bufferBuilder.method_22918(matrix4f, ry, rh, 0).method_22913(ru1, rv1).method_1344();
        bufferBuilder.method_22918(matrix4f, ry, y, 0).method_22913(ru1, rv0).method_1344();
        bufferBuilder.method_22918(matrix4f, x, y, 0).method_22913(ru0, rv0).method_1344();
        class_286.method_43433(bufferBuilder.method_1326());
    }

    /**
     * 指定済みテクスチャとシェーダーで描画
     * テクスチャサイズは256x256
     *
     * @param poseStack PoseStack
     * @param x         X
     * @param y         Y
     * @param ru0       テクスチャの開始地点X
     * @param rv0       テクスチャの開始地点Y
     * @param u1        相対テクスチャの終了地点X
     * @param v1        相対テクスチャの終了地点Y
     */
    public static void blitFloat(@NotNull class_4587 poseStack, float x, float y, float ru0, float rv0, float u1, float v1) {
        blitFloat(poseStack, x, y, ru0, rv0, u1, v1, 256, 256);
    }

    /**
     * GUI上を塗りつぶす
     * intの場合はGuiComponent.fillを推奨
     *
     * @param poseStack PoseStack
     * @param x         X
     * @param y         Y
     * @param width     幅
     * @param height    高さ
     * @param color     塗りつぶし色
     */
    public static void drawFill(@NotNull class_4587 poseStack, float x, float y, float width, float height, int color) {
        innerFill(poseStack.method_23760().method_23761(), x, y, width, height, color);
    }

    private static void innerFill(Matrix4f matrix4f, float x, float y, float w, float h, int coler) {
        float n;
        if (x < w) {
            n = x;
            x = w;
            w = n;
        }

        if (y < h) {
            n = y;
            y = h;
            h = n;
        }

        float a = (float) (coler >> 24 & 255) / 255.0F;
        float r = (float) (coler >> 16 & 255) / 255.0F;
        float g = (float) (coler >> 8 & 255) / 255.0F;
        float b = (float) (coler & 255) / 255.0F;

        class_287 bufferBuilder = class_289.method_1348().method_1349();
        RenderSystem.enableBlend();
        RenderSystem.disableTexture();
        RenderSystem.defaultBlendFunc();
        RenderSystem.setShader(class_757::method_34540);
        bufferBuilder.method_1328(class_293.class_5596.field_27382, class_290.field_1576);
        bufferBuilder.method_22918(matrix4f, x, h, 0.0F).method_22915(r, g, b, a).method_1344();
        bufferBuilder.method_22918(matrix4f, w, h, 0.0F).method_22915(r, g, b, a).method_1344();
        bufferBuilder.method_22918(matrix4f, w, y, 0.0F).method_22915(r, g, b, a).method_1344();
        bufferBuilder.method_22918(matrix4f, x, y, 0.0F).method_22915(r, g, b, a).method_1344();
        class_286.method_43433(bufferBuilder.method_1326());
        RenderSystem.enableTexture();
        RenderSystem.disableBlend();
    }

    /**
     * モデルを描画する
     *
     * @param poseStack       PoseStack
     * @param vertexConsumer  VertexConsumer
     * @param bakedModel      BakedModel
     * @param combinedLight   CombinedLight
     * @param combinedOverlay CombinedOverlay
     */
    public static void renderModel(class_4587 poseStack, class_4588 vertexConsumer, @NotNull class_1087 bakedModel, int combinedLight, int combinedOverlay) {
        Objects.requireNonNull(bakedModel);
        var bmr = mc.method_1541().method_3350();
        bmr.method_3367(poseStack.method_23760(), vertexConsumer, null, bakedModel, 1.0F, 1.0F, 1.0F, combinedLight, combinedOverlay);
    }

    /**
     * モデルを描画する
     *
     * @param poseStack       PoseStack
     * @param vertexConsumer  VertexConsumer
     * @param bakedModel      BakedModel
     * @param combinedLight   CombinedLight
     * @param combinedOverlay CombinedOverlay
     * @param color           色
     */
    public static void renderModel(class_4587 poseStack, class_4588 vertexConsumer, @NotNull class_1087 bakedModel, int combinedLight, int combinedOverlay, int color) {
        Objects.requireNonNull(bakedModel);
        var bmr = mc.method_1541().method_3350();
        float r = (float) (color >> 16 & 255) / 255.0F;
        float g = (float) (color >> 8 & 255) / 255.0F;
        float b = (float) (color & 255) / 255.0F;
        bmr.method_3367(poseStack.method_23760(), vertexConsumer, null, bakedModel, r, g, b, combinedLight, combinedOverlay);
    }

    @Deprecated
    public static void renderTextureSprite(class_2960 location, class_4587 poseStack, class_4597 multiBufferSource, float x, float y, float z, float pitch, float yaw, float roll, float width, float height, float u0, float v0, float u1, float v1, float textureWidth, float textureHeight, int combinedLightIn, int combinedOverlayIn) {
        poseStack.method_22903();
        poseStack.method_46416(x, y, z);
        poseRotateY(poseStack, yaw);
        poseRotateX(poseStack, pitch);
        poseRotateZ(poseStack, roll);
        renderTextureSprite(location, poseStack, multiBufferSource, width, height, u0, v0, u1, v1, textureWidth, textureHeight, combinedLightIn, combinedOverlayIn);
        poseStack.method_22909();
    }


    public static void renderTextureSprite(class_2960 location, class_4587 poseStack, class_4597 multiBufferSource, float width, float height, float u0, float v0, float u1, float v1, float textureWidth, float textureHeight, int combinedLightIn, int combinedOverlayIn) {
        renderSprite(poseStack, multiBufferSource.getBuffer(OERenderTypes.simpleSpriteCutout(location)), width, height, u0, v0, u1, v1, textureWidth, textureHeight, combinedLightIn, combinedOverlayIn);
    }

    public static void renderColorfulTextureSprite(class_2960 location, class_4587 poseStack, class_4597 multiBufferSource, float width, float height, float u0, float v0, float u1, float v1, float textureWidth, float textureHeight, int color, int combinedLightIn, int combinedOverlayIn) {
        renderColorfulSprite(poseStack, multiBufferSource.getBuffer(OERenderTypes.simpleSpriteCutout(location)), width, height, u0, v0, u1, v1, textureWidth, textureHeight, color, combinedLightIn, combinedOverlayIn);
    }

    public static void renderSprite(class_4587 poseStack, class_4588 vertexConsumer, float width, float height, float u0, float v0, float u1, float v1, float textureWidth, float textureHeight, int combinedLightIn, int combinedOverlayIn) {
        renderColorfulSprite(poseStack, vertexConsumer, width, height, u0, v0, u1, v1, textureWidth, textureHeight, 0xFFFFFFFF, combinedLightIn, combinedOverlayIn);
    }

    public static void renderColorfulSprite(class_4587 poseStack, class_4588 vertexConsumer, float width, float height, float u0, float v0, float u1, float v1, float textureWidth, float textureHeight, int color, int combinedLightIn, int combinedOverlayIn) {
        float wst = u0 / textureWidth;
        float wft = u1 / textureWidth + wst;
        float hst = v0 / textureHeight;
        float hft = v1 / textureHeight + hst;

        float a = (float) (color >> 24 & 255) / 255.0F;
        float r = (float) (color >> 16 & 255) / 255.0F;
        float g = (float) (color >> 8 & 255) / 255.0F;
        float b = (float) (color & 255) / 255.0F;

        class_4587.class_4665 pose = poseStack.method_23760();
        vertex(vertexConsumer, pose, 0, 0, 0, wst, hft, r, g, b, a, combinedOverlayIn, combinedLightIn);
        vertex(vertexConsumer, pose, width, 0, 0, wft, hft, r, g, b, a, combinedOverlayIn, combinedLightIn);
        vertex(vertexConsumer, pose, width, height, 0, wft, hst, r, g, b, a, combinedOverlayIn, combinedLightIn);
        vertex(vertexConsumer, pose, 0, height, 0, wst, hst, r, g, b, a, combinedOverlayIn, combinedLightIn);
    }

    private static void vertex(class_4588 builder, class_4587.class_4665 pose, float x, float y, float z, float u, float v, float r, float g, float b, float a, int combinedOverlayIn, int combinedLightIn) {
        builder.method_22918(pose.method_23761(), x, y, z).method_22915(r, g, b, a).method_22913(u, v).method_22922(combinedOverlayIn).method_22916(combinedLightIn).method_23763(pose.method_23762(), 0f, 0f, 0f).method_1344();
    }

    public static void posePlayerArm(class_4587 poseStack, class_1306 arm, float swingProgress, float equipProgress) {
        boolean bl = arm != class_1306.field_6182;
        float h = bl ? 1.0F : -1.0F;
        float j = class_3532.method_15355(swingProgress);
        float k = -0.3F * class_3532.method_15374(j * class_3532.field_29844);
        float l = 0.4F * class_3532.method_15374(j * class_3532.field_29846);
        float m = -0.4F * class_3532.method_15374(swingProgress * class_3532.field_29844);

        poseStack.method_46416(h * (k + 0.64000005F), l + -0.6F + equipProgress * -0.6F, m + -0.71999997F);
        poseStack.method_22907(class_7833.field_40716.rotationDegrees(h * 45.0F));
        float n = class_3532.method_15374(swingProgress * swingProgress * class_3532.field_29844);
        float o = class_3532.method_15374(j * class_3532.field_29844);
        poseStack.method_22907(class_7833.field_40716.rotationDegrees(h * o * 70.0F));
        poseStack.method_22907(class_7833.field_40718.rotationDegrees(h * n * -20.0F));
        poseStack.method_22904(h * -1.0F, 3.5999999046325684D, 3.5D);
        poseStack.method_22907(class_7833.field_40718.rotationDegrees(h * 120.0F));
        poseStack.method_22907(class_7833.field_40714.rotationDegrees(200.0F));
        poseStack.method_22907(class_7833.field_40716.rotationDegrees(h * -135.0F));
        poseStack.method_22904(h * 5.6F, 0.0D, 0.0D);
    }

    public static void renderPlayerArm(class_4587 poseStack, class_4597 multiBufferSource, class_1306 arm, int light) {
        if (mc.field_1724.method_5767()) return;
        boolean bl = arm != class_1306.field_6182;
        var pr = (class_1007) mc.method_1561().method_3953(mc.field_1724);
        RenderSystem.setShaderTexture(0, mc.field_1724.method_3117());
        if (bl) {
            pr.method_4220(poseStack, multiBufferSource, light, mc.field_1724);
        } else {
            pr.method_4221(poseStack, multiBufferSource, light, mc.field_1724);
        }
    }

    public static void poseHandItem(class_4587 poseStack, class_1306 arm, float swingProgress, float equipProgress) {
        boolean handFlg = arm == class_1306.field_6183;
        float s = -0.4F * class_3532.method_15374(class_3532.method_15355(swingProgress) * class_3532.field_29844);
        float r = 0.2F * class_3532.method_15374(class_3532.method_15355(swingProgress) * class_3532.field_29846);
        float l = -0.2F * class_3532.method_15374(swingProgress * class_3532.field_29844);
        int t = handFlg ? 1 : -1;
        poseStack.method_46416((float) t * s, r, l);
        poseStack.method_22904((float) t * 0.56F, -0.52F + equipProgress * -0.6F, -0.7200000286102295D);
        float g = class_3532.method_15374(swingProgress * swingProgress * class_3532.field_29844);
        poseRotateY(poseStack, (float) t * (45.0F + g * -20.0F));
        float h = class_3532.method_15374(class_3532.method_15355(swingProgress) * class_3532.field_29844);
        poseRotateZ(poseStack, (float) t * h * -20.0F);
        poseRotateX(poseStack, h * -80.0F);
        poseRotateY(poseStack, (float) t * -45.0F);
    }

    public static void renderHandItem(class_4587 poseStack, class_4597 multiBufferSource, class_1306 arm, class_1799 stack, int light) {
        boolean handFlg = arm == class_1306.field_6183;
        mc.field_1773.field_4012.method_3233(mc.field_1724, stack, handFlg ? class_809.class_811.field_4322 : class_809.class_811.field_4321, !handFlg, poseStack, multiBufferSource, light);
    }

    /**
     * 中央ぞろえで文字描画
     *
     * @param poseStack PoseStack
     * @param text      文字
     * @param x         中央X
     * @param y         Y
     * @param color     色(ARGB)
     * @since 2.0
     */
    public static void drawCenterFont(class_4587 poseStack, class_2561 text, float x, float y, int color) {
        mc.field_1772.method_30883(poseStack, text, x - ((float) mc.field_1772.method_27525(text) / 2f), y, color);
    }

    /**
     * 中央ぞろえ文字描画
     *
     * @param poseStack PoseStack
     * @param str       文字
     * @param x         中央X
     * @param y         Y
     * @param color     色(ARGB)
     * @since 2.0
     */
    public static void drawCenterFont(class_4587 poseStack, String str, float x, float y, int color) {
        mc.field_1772.method_1729(poseStack, str, x - ((float) mc.field_1772.method_1727(str) / 2f), y, color);
    }

    /**
     * 文字のスプライトを描画する
     *
     * @param text              テキスト
     * @param x                 X
     * @param y                 Y
     * @param color             色
     * @param shadow            影をつけるかどうか
     * @param lastPose          pose
     * @param multiBufferSource multiBufferSource
     * @param seeThrough        透けて見えるかどうか
     * @param bakedGlyphColor   背景色
     * @param packedLightCoords light
     * @return size?
     */
    public static int fontDrawInBatch(class_2561 text, float x, float y, int color, boolean shadow, Matrix4f lastPose, class_4597 multiBufferSource, boolean seeThrough, int bakedGlyphColor, int packedLightCoords) {
        return mc.field_1772.method_30882(text, x, y, color, shadow, lastPose, multiBufferSource, seeThrough, bakedGlyphColor, packedLightCoords);
    }

    /**
     * 文字のスプライトを描画する
     *
     * @param text              テキスト
     * @param x                 X
     * @param y                 Y
     * @param color             色
     * @param shadow            影をつけるかどうか
     * @param lastPose          pose
     * @param multiBufferSource multiBufferSource
     * @param seeThrough        透けて見えるかどうか
     * @param bakedGlyphColor   背景色
     * @param packedLightCoords light
     * @return size?
     */
    public static int fontDrawInBatch(String text, float x, float y, int color, boolean shadow, Matrix4f lastPose, class_4597 multiBufferSource, boolean seeThrough, int bakedGlyphColor, int packedLightCoords) {
        return mc.field_1772.method_27521(text, x, y, color, shadow, lastPose, multiBufferSource, seeThrough, bakedGlyphColor, packedLightCoords);
    }

    @Deprecated
    public static void renderTextSprite(class_4587 poseStack, class_4597 multiBufferSource, class_2561 text, float x, float y, float z, float size, float textX, float textY, int color, int combinedLightIn) {
        poseStack.method_22903();
        poseStack.method_46416(x, y, z);
        poseStack.method_22905(0.010416667F * size, -0.010416667F * size, 0.010416667F * size);
        mc.field_1772.method_30882(text, textX, -mc.field_1772.field_2000 + textY, color, false, poseStack.method_23760().method_23761(), multiBufferSource, false, 0, combinedLightIn);
        poseStack.method_22909();
    }

    @Deprecated
    public static void renderTextSprite(class_4587 poseStack, class_4597 multiBufferSource, class_2561 text, float x, float y, float z, float size, float textX, float textY, int combinedLightIn) {
        poseStack.method_22903();
        poseStack.method_46416(x, y, z);
        poseStack.method_22905(0.010416667F * size, -0.010416667F * size, 0.010416667F * size);
        mc.field_1772.method_30882(text, textX, -mc.field_1772.field_2000 + textY, 0, false, poseStack.method_23760().method_23761(), multiBufferSource, false, 0, combinedLightIn);
        poseStack.method_22909();
    }

    @Deprecated
    public static void renderCenterTextSprite(class_4587 poseStack, class_4597 multiBufferSource, class_2561 text, float x, float y, float z, float size, float textX, float textY, int color, int combinedLightIn) {
        poseStack.method_22903();
        poseStack.method_46416(x, y, z);
        poseStack.method_22905(0.010416667F * size, -0.010416667F * size, 0.010416667F * size);
        mc.field_1772.method_30882(text, ((float) -mc.field_1772.method_27525(text) / 2f) + textX, -mc.field_1772.field_2000 + textY, color, false, poseStack.method_23760().method_23761(), multiBufferSource, false, 0, combinedLightIn);
        poseStack.method_22909();
    }

    @Deprecated
    public static void renderCenterTextSprite(class_4587 poseStack, class_4597 multiBufferSource, class_2561 text, float x, float y, float z, float size, float textX, float textY, int combinedLightIn) {
        poseStack.method_22903();
        poseStack.method_46416(x, y, z);
        poseStack.method_22905(0.010416667F * size, -0.010416667F * size, 0.010416667F * size);
        mc.field_1772.method_30882(text, ((float) -mc.field_1772.method_27525(text) / 2f) + textX, -mc.field_1772.field_2000 + textY, 0, false, poseStack.method_23760().method_23761(), multiBufferSource, false, 0, combinedLightIn);
        poseStack.method_22909();
    }

    /**
     * 文字のスプライトを描画する
     *
     * @param text              テキスト
     * @param x                 X
     * @param y                 Y
     * @param color             色
     * @param shadow            影をつけるかどうか
     * @param poseStack         pose
     * @param multiBufferSource multiBufferSource
     * @param seeThrough        透けて見えるかどうか
     * @param bakedGlyphColor   背景色
     * @param packedLightCoords light
     */
    public static void renderFontSprite(class_2561 text, int x, int y, int color, boolean shadow, class_4587 poseStack, class_4597 multiBufferSource, boolean seeThrough, int bakedGlyphColor, int packedLightCoords) {
        poseStack.method_22903();
        poseStack.method_22905(-0.025F, -0.025F, 0.025F);
        Matrix4f matrix4f = poseStack.method_23760().method_23761();
        fontDrawInBatch(text, x, y, color, shadow, matrix4f, multiBufferSource, seeThrough, bakedGlyphColor, packedLightCoords);
        poseStack.method_22909();
    }

    /**
     * 文字のスプライトを描画する
     *
     * @param text              テキスト
     * @param x                 X
     * @param y                 Y
     * @param color             色
     * @param shadow            影をつけるかどうか
     * @param poseStack         pose
     * @param multiBufferSource multiBufferSource
     * @param seeThrough        透けて見えるかどうか
     * @param bakedGlyphColor   背景色
     * @param packedLightCoords light
     */
    public static void renderFontSprite(String text, int x, int y, int color, boolean shadow, class_4587 poseStack, class_4597 multiBufferSource, boolean seeThrough, int bakedGlyphColor, int packedLightCoords) {
        poseStack.method_22903();
        poseStack.method_22905(-0.025F, -0.025F, 0.025F);
        Matrix4f matrix4f = poseStack.method_23760().method_23761();
        fontDrawInBatch(text, x, y, color, shadow, matrix4f, multiBufferSource, seeThrough, bakedGlyphColor, packedLightCoords);
        poseStack.method_22909();
    }

    /**
     * 文字のスプライトを描画する
     *
     * @param text              テキスト
     * @param x                 X
     * @param y                 Y
     * @param color             色
     * @param shadow            影をつけるかどうか
     * @param poseStack         pose
     * @param multiBufferSource multiBufferSource
     * @param seeThrough        透けて見えるかどうか
     * @param bakedGlyphColor   背景色
     * @param packedLightCoords light
     */
    public static void renderCenterFontSprite(class_2561 text, int x, int y, int color, boolean shadow, class_4587 poseStack, class_4597 multiBufferSource, boolean seeThrough, int bakedGlyphColor, int packedLightCoords) {
        poseStack.method_22903();
        poseStack.method_22905(-0.025F, -0.025F, 0.025F);
        Matrix4f matrix4f = poseStack.method_23760().method_23761();
        x += (float) (-mc.field_1772.method_27525(text) / 2);
        fontDrawInBatch(text, x, y, color, shadow, matrix4f, multiBufferSource, seeThrough, bakedGlyphColor, packedLightCoords);
        poseStack.method_22909();
    }


    /**
     * 文字のスプライトを描画する
     *
     * @param text              テキスト
     * @param x                 X
     * @param y                 Y
     * @param color             色
     * @param shadow            影をつけるかどうか
     * @param poseStack         pose
     * @param multiBufferSource multiBufferSource
     * @param seeThrough        透けて見えるかどうか
     * @param bakedGlyphColor   背景色
     * @param packedLightCoords light
     */
    public static void renderCenterFontSprite(String text, int x, int y, int color, boolean shadow, class_4587 poseStack, class_4597 multiBufferSource, boolean seeThrough, int bakedGlyphColor, int packedLightCoords) {
        poseStack.method_22903();
        poseStack.method_22905(-0.025F, -0.025F, 0.025F);
        Matrix4f matrix4f = poseStack.method_23760().method_23761();
        x += (float) (-mc.field_1772.method_1727(text) / 2);
        fontDrawInBatch(text, x, y, color, shadow, matrix4f, multiBufferSource, seeThrough, bakedGlyphColor, packedLightCoords);
        poseStack.method_22909();
    }

    /**
     * 幅を固定して文字を描画
     * 幅に入りきらないと縮小し描画される
     *
     * @param poseStack PoseStack
     * @param text      文字
     * @param x         中央X
     * @param y         Y
     * @param color     色(ARGB)
     * @param width     幅
     */
    public static void drawFixedWidthFont(class_4587 poseStack, class_2561 text, float x, float y, int color, float width) {
        int size = mc.field_1772.method_27525(text);
        poseStack.method_22903();
        if (size > width) {
            float scale = width / size;
            x /= scale;
            y /= scale;
            poseScaleAll(poseStack, scale);
        }
        mc.field_1772.method_30883(poseStack, text, x, y, color);
        poseStack.method_22909();
    }

    /**
     * 幅を固定して文字を描画
     * 幅に入りきらないと縮小し描画される
     *
     * @param poseStack PoseStack
     * @param text      文字
     * @param x         中央X
     * @param y         Y
     * @param color     色(ARGB)
     * @param width     幅
     */
    public static void drawFixedWidthFont(class_4587 poseStack, String text, float x, float y, int color, float width) {
        int size = mc.field_1772.method_1727(text);
        poseStack.method_22903();
        if (size > width) {
            float scale = width / size;
            x /= scale;
            y /= scale;
            poseScaleAll(poseStack, scale);
        }
        mc.field_1772.method_1729(poseStack, text, x, y, color);
        poseStack.method_22909();
    }

    /**
     * GUI上でUUIDから取得したプレイヤーの顔を描画する
     *
     * @param poseStack PoseStack
     * @param uuid      プレイヤーUUID
     * @param x         X
     * @param y         Y
     * @since 2.0
     */
    public static void drawPlayerFace(class_4587 poseStack, UUID uuid, float x, float y) {
        drawPlayerFace(poseStack, uuid, x, y, 8);
    }

    /**
     * GUI上でUUIDから取得したプレイヤーの顔を描画する
     * サイズ変更可
     *
     * @param poseStack PoseStack
     * @param uuid      プレイヤーUUID
     * @param x         X
     * @param y         Y
     * @param size      サイズ
     * @since 2.0
     */
    public static void drawPlayerFace(class_4587 poseStack, UUID uuid, float x, float y, float size) {
        poseStack.method_22903();
        float sc = size / 8f;
        class_2960 plskin = OETextureUtils.getPlayerSkinTexture(uuid);
        drawTexture(plskin, poseStack, x, y, 8f * sc, 8f * sc, 8f * sc, 8f * sc, 64f * sc, 64f * sc);
        drawTexture(plskin, poseStack, x, y, 40f * sc, 8f * sc, 8f * sc, 8f * sc, 64f * sc, 64f * sc);
        poseStack.method_22909();
    }

    /**
     * GUI上で名前から取得したプレイヤーの顔を描画する
     *
     * @param poseStack PoseStack
     * @param name      プレイヤー名
     * @param x         X
     * @param y         Y
     */
    public static void drawPlayerFace(class_4587 poseStack, String name, float x, float y) {
        drawPlayerFace(poseStack, name, x, y, 8);
    }

    /**
     * GUI上で名前から取得したプレイヤーの顔を描画する
     * サイズ変更可
     *
     * @param poseStack PoseStack
     * @param name      プレイヤー名
     * @param x         X
     * @param y         Y
     * @param size      サイズ
     * @since 2.0
     */
    public static void drawPlayerFace(class_4587 poseStack, String name, float x, float y, float size) {
        poseStack.method_22903();
        float sc = size / 8f;
        class_2960 plskin = OETextureUtils.getPlayerSkinTexture(name);
        drawTexture(plskin, poseStack, x, y, 8f * sc, 8f * sc, 8f * sc, 8f * sc, 64f * sc, 64f * sc);
        drawTexture(plskin, poseStack, x, y, 40f * sc, 8f * sc, 8f * sc, 8f * sc, 64f * sc, 64f * sc);
        poseStack.method_22909();
    }

    /**
     * UUIDから取得したプレイヤーの顔スプライトを描画する
     *
     * @param poseStack         PoseStack
     * @param multiBufferSource MultiBufferSource
     * @param uuid              プレイヤーUUID
     * @param size              サイズ
     * @param combinedLightIn   CombinedLightIn
     * @param combinedOverlayIn CombinedOverlayIn
     */
    public static void renderPlayerFaceSprite(class_4587 poseStack, class_4597 multiBufferSource, UUID uuid, float size, int combinedLightIn, int combinedOverlayIn) {
        renderPlayerFaceSprite(poseStack, multiBufferSource.getBuffer(OERenderTypes.simpleSpriteCutout(OETextureUtils.getPlayerSkinTexture(uuid))), size, combinedLightIn, combinedOverlayIn);
    }

    /**
     * UUIDから取得したプレイヤーの顔スプライトを描画する
     *
     * @param poseStack         PoseStack
     * @param multiBufferSource MultiBufferSource
     * @param name              プレイヤー名
     * @param size              サイズ
     * @param combinedLightIn   CombinedLightIn
     * @param combinedOverlayIn CombinedOverlayIn
     */
    public static void renderPlayerFaceSprite(class_4587 poseStack, class_4597 multiBufferSource, String name, float size, int combinedLightIn, int combinedOverlayIn) {
        renderPlayerFaceSprite(poseStack, multiBufferSource.getBuffer(OERenderTypes.simpleSpriteCutout(OETextureUtils.getPlayerSkinTexture(name))), size, combinedLightIn, combinedOverlayIn);
    }

    /**
     * プレイヤーの顔スプライトを描画する
     *
     * @param poseStack         PoseStack
     * @param vertexConsumer    VertexConsumer
     * @param size              サイズ
     * @param combinedLightIn   CombinedLightIn
     * @param combinedOverlayIn CombinedOverlayIn
     */
    public static void renderPlayerFaceSprite(class_4587 poseStack, class_4588 vertexConsumer, float size, int combinedLightIn, int combinedOverlayIn) {
        poseStack.method_22903();
        float sc = size / 8f;
        renderSprite(poseStack, vertexConsumer, size, size, 8f * sc, 8f * sc, 8f * sc, 8f * sc, 64f * sc, 64f * sc, combinedLightIn, combinedOverlayIn);

        poseStack.method_22903();
        poseStack.method_46416(0, 0, class_3532.field_29849);
        renderSprite(poseStack, vertexConsumer, size, size, 40f * sc, 8f * sc, 8f * sc, 8f * sc, 64f * sc, 64f * sc, combinedLightIn, combinedOverlayIn);
        poseStack.method_22909();

        poseStack.method_22909();
    }

    public static float getPartialTicks() {
        return OEClientExpectPlatform.getPartialTicks();
    }

    public static void noTransAndRotModelPart(Runnable runnable) {
        SKIP_TRANSANDROT_MODELPART.set(true);
        runnable.run();
        SKIP_TRANSANDROT_MODELPART.set(false);
    }

    public static void renderPlayerArmNoTransAndRot(class_4587 poseStack, class_4597 multiBufferSource, class_1306 arm, int light) {
        noTransAndRotModelPart(() -> renderPlayerArm(poseStack, multiBufferSource, arm, light));
    }
}
