package dev.felnull.otyacraftengine.data.model;

import net.minecraft.class_2248;
import net.minecraft.class_2960;
import net.minecraft.class_4917;
import org.jetbrains.annotations.NotNull;

/**
 * それぞれのプラットフォームで同じ動作をするブロックステータスとブロックモデル生成
 */
public interface BlockStateAndModelProviderAccess {
    /**
     * 簡素な立方体ブロックのステート、ブロックモデル、アイテムモデルを生成する
     *
     * @param block ブロック
     */
    void simpleCubeBlockStateModelAndItemModel(@NotNull class_2248 block);

    /**
     * 全面同じテクスチャのブロックモデルを生成
     *
     * @param blockLocation   ブロックロケーション
     * @param textureLocation テクスチャロケーション
     * @return モデル
     */
    @NotNull
    default FileModel cubeAllBlockModel(@NotNull class_2960 blockLocation, @NotNull class_2960 textureLocation) {
        return cubeAllBlockModel(blockLocation, FileTexture.of(textureLocation));
    }

    /**
     * 全面同じテクスチャのブロックモデルを生成
     *
     * @param blockLocation ブロックロケーション
     * @param fileTexture   ファイルテクスチャ
     * @return モデル
     */
    @NotNull
    FileModel cubeAllBlockModel(@NotNull class_2960 blockLocation, @NotNull FileTexture fileTexture);

    /**
     * 全面同じテクスチャのブロックモデルを生成
     *
     * @param block           ブロック
     * @param textureLocation テクスチャロケーション
     * @return モデル
     */
    @NotNull
    default FileModel cubeAllBlockModel(@NotNull class_2248 block, @NotNull class_2960 textureLocation) {
        return cubeAllBlockModel(block, FileTexture.of(textureLocation));
    }

    /**
     * 全面同じテクスチャのブロックモデルを生成
     *
     * @param block       ブロック
     * @param fileTexture ファイルテクスチャ
     * @return モデル
     */
    @NotNull
    FileModel cubeAllBlockModel(@NotNull class_2248 block, @NotNull FileTexture fileTexture);

    /**
     * それぞれの方向で違うテクスチャを指定する
     *
     * @param blockLocation ブロックロケーション
     * @param down          下側のテクスチャ
     * @param up            上側のテクスチャ
     * @param north         北側のテクスチャ
     * @param south         南側のテクスチャ
     * @param east          東側のテクスチャ
     * @param west          西側のテクスチャ
     * @return モデル
     */
    @NotNull
    FileModel cubeBlockModel(@NotNull class_2960 blockLocation, @NotNull FileTexture down, @NotNull FileTexture up, @NotNull FileTexture north, @NotNull FileTexture south, @NotNull FileTexture east, @NotNull FileTexture west);

    /**
     * それぞれの方向で違うテクスチャを指定する
     *
     * @param blockLocation ブロックロケーション
     * @param down          下側のテクスチャ
     * @param up            上側のテクスチャ
     * @param north         北側のテクスチャ
     * @param south         南側のテクスチャ
     * @param east          東側のテクスチャ
     * @param west          西側のテクスチャ
     * @return モデル
     */
    @NotNull
    default FileModel cubeBlockModel(@NotNull class_2960 blockLocation, @NotNull class_2960 down, @NotNull class_2960 up, @NotNull class_2960 north, @NotNull class_2960 south, @NotNull class_2960 east, @NotNull class_2960 west) {
        return cubeBlockModel(blockLocation, FileTexture.of(down), FileTexture.of(up), FileTexture.of(north), FileTexture.of(south), FileTexture.of(east), FileTexture.of(west));
    }

    /**
     * それぞれの方向で違うテクスチャを指定する
     *
     * @param block ブロック
     * @param down  下側のテクスチャ
     * @param up    上側のテクスチャ
     * @param north 北側のテクスチャ
     * @param south 南側のテクスチャ
     * @param east  東側のテクスチャ
     * @param west  西側のテクスチャ
     * @return モデル
     */
    @NotNull
    default FileModel cubeBlockModel(@NotNull class_2248 block, @NotNull class_2960 down, @NotNull class_2960 up, @NotNull class_2960 north, @NotNull class_2960 south, @NotNull class_2960 east, @NotNull class_2960 west) {
        return cubeBlockModel(block, FileTexture.of(down), FileTexture.of(up), FileTexture.of(north), FileTexture.of(south), FileTexture.of(east), FileTexture.of(west));
    }

    /**
     * それぞれの方向で違うテクスチャを指定する
     *
     * @param block ブロック
     * @param down  下側のテクスチャ
     * @param up    上側のテクスチャ
     * @param north 北側のテクスチャ
     * @param south 南側のテクスチャ
     * @param east  東側のテクスチャ
     * @param west  西側のテクスチャ
     * @return モデル
     */
    @NotNull
    FileModel cubeBlockModel(@NotNull class_2248 block, @NotNull FileTexture down, @NotNull FileTexture up, @NotNull FileTexture north, @NotNull FileTexture south, @NotNull FileTexture east, @NotNull FileTexture west);

    /**
     * 上下と側面ごとに違うテクスチャを指定する
     *
     * @param blockLocation ブロックロケーション
     * @param bottom        下側のテクスチャ
     * @param side          側面のテクスチャ
     * @param top           上側のテクスチャ
     * @return モデル
     */
    @NotNull
    default FileModel cubeBottomTopBlockModel(@NotNull class_2960 blockLocation, @NotNull class_2960 bottom, @NotNull class_2960 side, @NotNull class_2960 top) {
        return cubeBottomTopBlockModel(blockLocation, FileTexture.of(bottom), FileTexture.of(side), FileTexture.of(top));
    }

    /**
     * 上下と側面ごとに違うテクスチャを指定する
     *
     * @param blockLocation ブロックロケーション
     * @param bottom        下側のテクスチャ
     * @param side          側面のテクスチャ
     * @param top           上側のテクスチャ
     * @return モデル
     */
    @NotNull
    FileModel cubeBottomTopBlockModel(@NotNull class_2960 blockLocation, @NotNull FileTexture bottom, @NotNull FileTexture side, @NotNull FileTexture top);

    /**
     * 上下と側面ごとに違うテクスチャを指定する
     *
     * @param block  ブロック
     * @param bottom 下側のテクスチャ
     * @param side   側面のテクスチャ
     * @param top    上側のテクスチャ
     * @return モデル
     */
    @NotNull
    default FileModel cubeBottomTopBlockModel(@NotNull class_2248 block, @NotNull class_2960 bottom, @NotNull class_2960 side, @NotNull class_2960 top) {
        return cubeBottomTopBlockModel(bottom, FileTexture.of(bottom), FileTexture.of(side), FileTexture.of(top));
    }

    /**
     * 上下と側面ごとに違うテクスチャを指定する
     *
     * @param block  ブロック
     * @param bottom 下側のテクスチャ
     * @param side   側面のテクスチャ
     * @param top    上側のテクスチャ
     * @return モデル
     */
    @NotNull
    FileModel cubeBottomTopBlockModel(@NotNull class_2248 block, @NotNull FileTexture bottom, @NotNull FileTexture side, @NotNull FileTexture top);

    /**
     * 親モデルを継承するブロックモデル
     *
     * @param block          ブロック
     * @param parentLocation 親モデルのロケーション
     * @return モデル
     */
    @NotNull
    MutableFileModel parentedBlockModel(@NotNull class_2248 block, @NotNull class_2960 parentLocation);

    /**
     * 既存のモデル
     *
     * @param location ロケーション
     * @return モデル
     */
    @NotNull
    FileModel existingModel(@NotNull class_2960 location);

    /**
     * パーティクルのみのモデル
     *
     * @param block            ブロック
     * @param particleLocation パーティクルのテクスチャロケーション
     * @return モデル
     */
    @NotNull
    default FileModel particleOnlyModel(@NotNull class_2248 block, @NotNull class_2960 particleLocation) {
        return particleOnlyModel(block, FileTexture.of(particleLocation));
    }

    @NotNull
    FileModel particleOnlyModel(@NotNull class_2248 block, @NotNull FileTexture particleLocation);

    /**
     * 指定のモデルが常時適用されるステート
     *
     * @param block ブロック
     * @param model モデル
     */
    void simpleBlockState(@NotNull class_2248 block, @NotNull FileModel model);

    /**
     * 指定のブロックモデルを継承するアイテムモデルを生成<br>
     * Fabricの場合、ステートを登録していればアイテムモデル未登録でも勝手に自動登録される。<br>
     * でもForgeでは自動登録されないのでこれを利用してください。
     *
     * @param block ブロック
     * @param model モデル
     */
    void simpleBlockItemModel(@NotNull class_2248 block, @NotNull FileModel model);

    /**
     * 方向によって回転するステート
     *
     * @param block ブロック
     * @param model モデル
     */
    void horizontalBlockState(@NotNull class_2248 block, @NotNull FileModel model);

    /**
     * ビルトインエンティティのアイテムモデル
     *
     * @param block ブロック
     */
    default void builtinEntityBlockItemModel(@NotNull class_2248 block) {
        parentedBlockItemModel(block, new class_2960("builtin/entity"));
    }

    /**
     * 指定のモデルロケーションを継承するアイテムもでるを生成
     *
     * @param block          ブロック
     * @param parentLocation 親モデルのロケーション
     */
    void parentedBlockItemModel(@NotNull class_2248 block, @NotNull class_2960 parentLocation);

    /**
     * ブロックステートジェネレーターを登録
     *
     * @param blockStateGenerator ブロックステートジェネレーター
     */
    void addBlockStateGenerator(@NotNull class_4917 blockStateGenerator);

    /**
     * アイテムのモデルを生成する場合に利用するアイテム生成へのアクセスを取得
     *
     * @return アイテム生成へのアクセス
     */
    ItemModelProviderAccess itemModelProviderAccess();
}
