package dev.felnull.specialmodelloader.impl.model.obj;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.gson.JsonObject;
import de.javagl.obj.*;
import dev.felnull.specialmodelloader.api.model.LoadedResource;
import dev.felnull.specialmodelloader.api.model.obj.ObjModelLoader;
import dev.felnull.specialmodelloader.api.model.obj.ObjModelOption;
import dev.felnull.specialmodelloader.impl.SpecialModelLoader;
import org.apache.commons.lang3.ArrayUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import java.io.BufferedReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import net.minecraft.class_1100;
import net.minecraft.class_2960;
import net.minecraft.class_3298;
import net.minecraft.class_3300;

public class ObjModelLoaderImp implements ObjModelLoader {
    public static final ObjModelLoaderImp INSTANCE = new ObjModelLoaderImp();

    @Override
    public @Nullable LoadedResource loadResource(@NotNull class_3300 resourceManager, @NotNull JsonObject modelJson) {
        if (!modelJson.has("model") || !modelJson.get("model").isJsonPrimitive() || !modelJson.getAsJsonPrimitive("model").isString()) {
            return null;
        }

        class_2960 modelLocation = class_2960.method_60654(modelJson.get("model").getAsString());

        return loadResource(resourceManager, modelLocation, ObjModelOption.parse(modelJson));
    }

    @Override
    public @Nullable LoadedResource loadResource(@NotNull class_3300 resourceManager, @NotNull class_2960 modelLocation, @NotNull ObjModelOption option) {
        Optional<class_3298> res = resourceManager.method_14486(modelLocation);

        if (res.isEmpty()) {
            return null;
        }

        try (var reader = res.get().method_43039()) {
            Obj obj = ObjUtils.convertToRenderable(ObjReader.read(reader));

            class_2960 mtlDirLoc;
            List<String> mtlFileNames;

            String mtlOverride;
            if ((mtlOverride = option.getMtlOverride()) != null) {
                String[] overrideSplit = mtlOverride.split("/");

                mtlDirLoc = class_2960.method_60654(String.join("/", ArrayUtils.remove(overrideSplit, overrideSplit.length - 1)));
                mtlFileNames = ImmutableList.of(overrideSplit[overrideSplit.length - 1]);
            } else {
                String[] mtlDirPaths = modelLocation.method_12832().split("/");
                mtlDirPaths = ArrayUtils.remove(mtlDirPaths, mtlDirPaths.length - 1);

                mtlDirLoc = class_2960.method_60655(modelLocation.method_12836(), String.join("/", mtlDirPaths));
                mtlFileNames = obj.getMtlFileNames();
            }

            return new ObjModelLoadedResource(modelLocation, obj,
                    ImmutableMap.copyOf(loadMtl(resourceManager, mtlDirLoc, mtlFileNames)), option);
        } catch (IOException e) {
            throw new IllegalStateException("Failed to load obj file.", e);
        }
    }

    @Override
    public @NotNull class_1100 makeModel(@NotNull LoadedResource loadedResource) {
        if (loadedResource instanceof ObjModelLoadedResource objRes) {
            return new ObjUnbakedModelModel(objRes.location(), objRes.obj(), objRes.mtl(), objRes.option());
        } else {
            throw new IllegalArgumentException("A loaded resource that is not an OBJ model was received as an argument.");
        }
    }

    private Map<String, Mtl> loadMtl(class_3300 resourceManager, class_2960 location, List<String> mtlNames) {
        return mtlNames.stream()
                .flatMap(r -> loadMtl(resourceManager, location, r).stream())
                .collect(Collectors.toMap(Mtl::getName, r -> r));
    }

    private List<Mtl> loadMtl(class_3300 resourceManager, class_2960 location, String mtlName) {
        var loc = class_2960.method_60655(location.method_12836(), location.method_12832() + "/" + mtlName);
        return resourceManager.method_14486(loc).map(res -> {
            try (var reader = res.method_43039()) {
                return MtlReader.read(reader);
            } catch (IOException e) {
                SpecialModelLoader.LOGGER.error("Failed to read mtl file.", e);
                return new ArrayList<Mtl>();
            }
        }).orElseGet(List::of);
    }

    @Override
    public String getId() {
        return "obj";
    }
}
