package dev.felnull.otyacraftengine.item;

import net.minecraft.core.NonNullList;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.world.Container;
import net.minecraft.world.ContainerHelper;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.item.ItemStack;

import java.util.function.Function;

public class ItemContainer implements Container {
    private final ItemStack itemStack;
    private final NonNullList<ItemStack> items;
    private final String tagName;
    private final Function<Player, Boolean> valid;

    public ItemContainer(ItemStack itemStack, int size, String tagName, Function<Player, Boolean> valid) {
        this.itemStack = itemStack;
        this.items = NonNullList.withSize(size, ItemStack.EMPTY);
        loadItemList(itemStack, items, tagName);
        this.tagName = tagName;
        this.valid = valid;
    }

    @Override
    public int getContainerSize() {
        return this.items.size();
    }

    @Override
    public boolean isEmpty() {
        return items.stream().allMatch(ItemStack::isEmpty);
    }

    @Override
    public ItemStack getItem(int i) {
        return items.get(i);
    }

    @Override
    public ItemStack removeItem(int i, int j) {
        var ret = ContainerHelper.removeItem(items, i, j);
        saveItems();
        return ret;
    }

    @Override
    public ItemStack removeItemNoUpdate(int i) {
        var ret = ContainerHelper.takeItem(items, i);
        saveItems();
        return ret;
    }

    @Override
    public void setItem(int i, ItemStack stack) {
        ItemStack itemstack = items.get(i);
        boolean flag = !stack.isEmpty() && stack.sameItem(itemstack) && ItemStack.tagMatches(stack, itemstack);
        items.set(i, stack);
        if (stack.getCount() > this.getMaxStackSize()) {
            stack.setCount(this.getMaxStackSize());
        }
        if (flag)
            this.setChanged();
    }

    @Override
    public void setChanged() {
        saveItems();
    }

    @Override
    public boolean stillValid(Player player) {
        return valid.apply(player);
    }

    @Override
    public void clearContent() {
        items.clear();
    }

    @Override
    public void stopOpen(Player player) {
        Container.super.stopOpen(player);
        saveItems();
    }

    public void saveItems() {
        saveItemList(itemStack, items, tagName);
    }

    public static void loadItemList(ItemStack itemStack, NonNullList<ItemStack> items, String tagName) {
        var tag = itemStack.getTag();
        if (tag != null)
            ContainerHelper.loadAllItems(tag.getCompound(tagName), items);
    }

    public static void saveItemList(ItemStack itemStack, NonNullList<ItemStack> items, String tagName) {
        var tag = itemStack.getOrCreateTag();
        if (!tag.contains(tagName))
            tag.put(tagName, new CompoundTag());
        ContainerHelper.saveAllItems(tag.getCompound(tagName), items);
    }

    public ItemStack getItemStack() {
        return itemStack;
    }
}
