package dev.felnull.otyacraftengine.client.util;

import TextureLoadResult;
import TextureScale;
import com.madgag.gif.fmsware.GifDecoder;
import com.mojang.authlib.minecraft.MinecraftProfileTexture;
import dev.felnull.fnjl.util.FNDataUtil;
import dev.felnull.fnjl.util.FNImageUtil;
import dev.felnull.fnjl.util.FNMath;
import dev.felnull.fnjl.util.FNURLUtil;
import dev.felnull.otyacraftengine.OtyacraftEngine;
import dev.felnull.otyacraftengine.client.renderer.texture.DynamicGifTexture;
import dev.felnull.otyacraftengine.impl.client.OEClientExpectPlatform;
import dev.felnull.otyacraftengine.util.OEPaths;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import record;
import var;
import java.io.*;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;
import net.minecraft.class_1011;
import net.minecraft.class_1043;
import net.minecraft.class_1047;
import net.minecraft.class_1068;
import net.minecraft.class_1657;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3300;

public class OETextureUtil {
    private static final class_310 mc = class_310.method_1551();
    protected static final Map<UUID, TextureLoadResult> NATIVE_TEXTURES = new HashMap<>();
    protected static final List<UUID> LOAD_TEXTURES = new ArrayList<>();
    protected static class_2960 LOADING_ICON;
    protected static final Map<String, String> URL_FILENAME_INDEX = new HashMap<>();
    protected static final Map<String, UUID> URL_TEXTURES_UUIDS = new HashMap<>();
    protected static final List<String> URL_LOAD_TEXTURES = new ArrayList<>();

    /**
     * ロード中アイコンを取得
     *
     * @return ロケーション
     */
    public static class_2960 getLoadingIcon() {
        if (LOADING_ICON == null) {
            class_3300 rm = mc.method_1478();
            try {
                LOADING_ICON = OETextureUtil.loadNativeTexture(rm.method_14486(new class_2960(OtyacraftEngine.MODID, "textures/gui/loading.gif")).method_14482(), class_1047.method_4539()).location();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return LOADING_ICON;
    }

    /**
     * UUIDからプレイヤースキンテクスチャ取得
     * 存在しない場合はnullを返す
     *
     * @param uuid プレイヤーUUID
     * @return プレイヤースキンテクスチャID
     * @since 2.0
     */
    public static class_2960 getPlayerSkinTexture(UUID uuid) {
        return getPlayerTexture(MinecraftProfileTexture.Type.SKIN, uuid);
    }

    /**
     * UUIDからプレイヤーマントテクスチャ取得
     * 存在しない場合はnullを返す
     *
     * @param uuid プレイヤーUUID
     * @return プレイヤーマントテクスチャID
     * @since 2.0
     */
    public static class_2960 getPlayerCapeTexture(UUID uuid) {
        return getPlayerTexture(MinecraftProfileTexture.Type.CAPE, uuid);
    }

    /**
     * UUIDからプレイヤーエリトラテクスチャ取得
     * 存在しない場合はnullを返す
     *
     * @param uuid プレイヤーUUID
     * @return プレイヤーエリトラテクスチャID
     * @since 2.0
     */
    public static class_2960 getPlayerElytraTexture(UUID uuid) {
        return getPlayerTexture(MinecraftProfileTexture.Type.ELYTRA, uuid);
    }

    /**
     * 名前からプレイヤースキンテクスチャ取得
     * 存在しない場合はnullを返す
     *
     * @param name プレイヤー名
     * @return プレイヤースキンテクスチャID
     */
    public static class_2960 getPlayerSkinTexture(String name) {
        return getPlayerTexture(MinecraftProfileTexture.Type.SKIN, name);
    }

    /**
     * 名前からプレイヤーマントテクスチャ取得
     * 存在しない場合はnullを返す
     *
     * @param name プレイヤー名
     * @return プレイヤーマントテクスチャID
     */
    public static class_2960 getPlayerCapeTexture(String name) {
        return getPlayerTexture(MinecraftProfileTexture.Type.CAPE, name);
    }

    /**
     * 名前からプレイヤーエリトラテクスチャ取得
     * 存在しない場合はnullを返す
     *
     * @param name プレイヤー名
     * @return プレイヤーエリトラテクスチャID
     */
    public static class_2960 getPlayerElytraTexture(String name) {
        return getPlayerTexture(MinecraftProfileTexture.Type.ELYTRA, name);
    }

    /**
     * 名前からプレイヤーテクスチャ取得
     * 存在しない場合はnullを返す
     *
     * @param type テクスチャタイプ
     * @param name プレイヤー名
     * @return プレイヤーテクスチャロケーション
     */
    @Nullable
    public static class_2960 getPlayerTexture(MinecraftProfileTexture.Type type, @NotNull String name) {
        if (mc.field_1724 != null) {
            var pl = mc.field_1724.field_3944.method_2874(name);
            if (pl != null)
                return switch (type) {
                    case SKIN -> pl.getSkinLocation();
                    case CAPE -> pl.getCapeLocation();
                    case ELYTRA -> pl.getElytraLocation();
                };
        }
        var gameProfile = OEClientUtil.getClientPlayerProfile(name);
        var tex = mc.method_1582().method_4654(gameProfile).get(type);
        return tex != null ? mc.method_1582().method_4656(tex, type) : type == MinecraftProfileTexture.Type.SKIN ? class_1068.method_4648(class_1657.method_7271(gameProfile)) : null;
    }

    /**
     * UUIDからプレイヤーテクスチャ取得
     * 存在しない場合はnullを返す
     *
     * @param type テクスチャタイプ
     * @param uuid プレイヤーUUID
     * @return プレイヤーテクスチャロケーション
     * @since 2.0
     */
    public static class_2960 getPlayerTexture(MinecraftProfileTexture.Type type, @NotNull UUID uuid) {
        if (mc.field_1724 != null) {
            var pl = mc.field_1724.field_3944.method_2871(uuid);
            if (pl != null)
                return switch (type) {
                    case SKIN -> pl.getSkinLocation();
                    case CAPE -> pl.getCapeLocation();
                    case ELYTRA -> pl.getElytraLocation();
                };
        }
        return OEClientUtil.getPlayerNameByUUID(uuid).map(n -> getPlayerTexture(type, n)).orElseGet(() -> type == MinecraftProfileTexture.Type.SKIN ? class_1068.method_4648(uuid) : null);
    }

    /**
     * 非同期でURLからネイティブテクスチャを取得する
     *
     * @param url  URL
     * @param cash キャッシュを取るかどうか
     * @return テクスチャID
     */
    public static class_2960 getURLTextureAsyncLoad(String url, boolean cash) {
        return getURLTextureAsyncLoad(url, cash, class_1047.method_4539());
    }

    /**
     * 非同期でURLからネイティブテクスチャを取得する
     *
     * @param url            URL
     * @param cash           キャッシュを取るかどうか
     * @param failureTexture 読み込み失敗時のテクスチャ
     * @return テクスチャID
     */
    public static class_2960 getURLTextureAsyncLoad(String url, boolean cash, class_2960 failureTexture) {
        return getURLTextureAsyncLoad(url, cash, getLoadingIcon(), failureTexture);
    }

    /**
     * 非同期でURLからネイティブテクスチャを取得する
     *
     * @param url            URL
     * @param cash           キャッシュを取るかどうか
     * @param loadTexture    ロード中のテクスチャ
     * @param failureTexture 読み込み失敗時のテクスチャ
     * @return テクスチャID
     */
    public static class_2960 getURLTextureAsyncLoad(String url, boolean cash, class_2960 loadTexture, class_2960 failureTexture) {
        if (URL_TEXTURES_UUIDS.containsKey(url))
            return getNativeTexture(URL_TEXTURES_UUIDS.get(url), null);

        if (URL_LOAD_TEXTURES.contains(url))
            return loadTexture;

        CompletableFuture.runAsync(() -> {
            URL_LOAD_TEXTURES.add(url);
            var tex = loadURLTexture(url, cash, failureTexture);
            mc.method_20493(() -> {
                UUID id = UUID.randomUUID();
                NATIVE_TEXTURES.put(id, tex);
                URL_LOAD_TEXTURES.remove(url);
                URL_TEXTURES_UUIDS.put(url, id);
            });
        });

        return loadTexture;
    }

    /**
     * URLからテクスチャ取得
     *
     * @param url  URL
     * @param cash キャッシュを取るかどうか
     * @return テクスチャID
     */
    public static class_2960 getURLTexture(String url, boolean cash) {
        return getURLTexture(url, cash, class_1047.method_4539());
    }

    /**
     * URLからテクスチャ取得
     *
     * @param url            URL
     * @param cash           キャッシュを取るかどうか
     * @param failureTexture 読み込み失敗時のテクスチャ
     * @return テクスチャID
     */
    public static class_2960 getURLTexture(String url, boolean cash, class_2960 failureTexture) {
        if (URL_TEXTURES_UUIDS.containsKey(url))
            return getNativeTexture(URL_TEXTURES_UUIDS.get(url), null);

        UUID id = UUID.randomUUID();
        TextureLoadResult tx = loadURLTexture(url, cash, failureTexture);
        URL_TEXTURES_UUIDS.put(url, id);
        NATIVE_TEXTURES.put(id, tx);
        return tx.location();
    }

    private static TextureLoadResult loadURLTexture(String url, boolean cash, class_2960 failureTexture) {
        try {
            InputStream stream;
            byte[] md5 = FNDataUtil.createMD5Hash(url.getBytes(StandardCharsets.UTF_8));
            String identification = Base64.getEncoder().encodeToString(md5);
            Path oep = OEPaths.getClientOEFolderPath().resolve("url_textures");
            if (URL_FILENAME_INDEX.containsKey(identification) && oep.resolve(URL_FILENAME_INDEX.get(identification)).toFile().exists()) {
                stream = new BufferedInputStream(new FileInputStream(oep.resolve(URL_FILENAME_INDEX.get(identification)).toFile()));
            } else {
                HttpURLConnection connection = FNURLUtil.getConnection(new URL(url));
                long length = connection.getContentLengthLong();
                long max = 1024L * 1024L * 3;
                if (length > max)
                    throw new IOException("Size Over: " + max + "byte" + " current: " + length + "byte");
                if (cash) {
                    byte[] data = connection.getInputStream().readAllBytes();
                    UUID uid = UUID.randomUUID();
                    Files.write(oep.resolve(uid.toString()), data);
                    URL_FILENAME_INDEX.put(identification, uid.toString());
                    stream = new ByteArrayInputStream(data);
                } else {
                    stream = connection.getInputStream();
                }
            }
            return loadNativeTexture(stream, failureTexture);
        } catch (Exception ex) {
            ex.printStackTrace();
        }
        return new TextureLoadResult(failureTexture, true);
    }

    /**
     * 非同期でネイティブテクスチャを取得する
     *
     * @param id     テクスチャ管理ID
     * @param stream リソース
     * @return テクスチャID
     */
    public static class_2960 getNativeTextureAsyncLoad(UUID id, InputStream stream) {
        return getNativeTextureAsyncLoad(id, stream, class_1047.method_4539());
    }

    /**
     * 非同期でネイティブテクスチャを取得する
     *
     * @param id             テクスチャ管理ID
     * @param stream         リソース
     * @param failureTexture 読み込み失敗時のテクスチャ
     * @return テクスチャID
     */
    public static class_2960 getNativeTextureAsyncLoad(UUID id, InputStream stream, class_2960 failureTexture) {
        return getNativeTextureAsyncLoad(id, stream, getLoadingIcon(), failureTexture);
    }

    /**
     * 非同期でネイティブテクスチャを取得する
     *
     * @param id             テクスチャ管理ID
     * @param stream         リソース
     * @param loadTexture    ロード中のテクスチャ
     * @param failureTexture 読み込み失敗時のテクスチャ
     * @return テクスチャID
     */
    public static class_2960 getNativeTextureAsyncLoad(UUID id, InputStream stream, class_2960 loadTexture, class_2960 failureTexture) {
        if (NATIVE_TEXTURES.containsKey(id))
            return NATIVE_TEXTURES.get(id).location();

        if (LOAD_TEXTURES.contains(id))
            return loadTexture;

        CompletableFuture.runAsync(() -> {
            LOAD_TEXTURES.add(id);
            var tex = loadNativeTexture(stream, failureTexture);
            mc.method_20493(() -> {
                NATIVE_TEXTURES.put(id, tex);
                LOAD_TEXTURES.remove(id);
            });
        });

        return loadTexture;
    }

    /**
     * テクスチャのスケールを取得
     *
     * @param location テクスチャロケーション
     * @return スケール
     */
    public static TextureScale getTextureScale(class_2960 location) {
        if (location != null && mc.method_1531().method_4619(location) instanceof class_1043 texture) {
            int w = texture.method_4525().method_4307();
            int h = texture.method_4525().method_4323();
            var sc = FNMath.scale(w, h);
            return new TextureScale(sc.getX(), sc.getY());
        }
        return null;
    }


    /**
     * 非同期でネイティブテクスチャを取得する
     *
     * @param id     テクスチャ管理ID
     * @param stream リソース
     * @return テクスチャID
     */
    public static class_2960 getNativeTexture(UUID id, InputStream stream) {
        return getNativeTexture(id, stream, class_1047.method_4539());
    }

    /**
     * 非同期でネイティブテクスチャを取得する
     *
     * @param id             テクスチャ管理ID
     * @param stream         リソース
     * @param failureTexture 読み込み失敗時のテクスチャ
     * @return テクスチャID
     */
    public static class_2960 getNativeTexture(UUID id, InputStream stream, class_2960 failureTexture) {
        if (NATIVE_TEXTURES.containsKey(id))
            return NATIVE_TEXTURES.get(id).location();

        var tex = loadNativeTexture(stream, failureTexture);
        NATIVE_TEXTURES.put(id, tex);
        return tex.location();
    }

    protected static TextureLoadResult loadNativeTexture(InputStream stream, class_2960 failureTexture) {
        try {
            byte[] data = stream.readAllBytes();
            GifDecoder decoder = new GifDecoder();
            boolean gifFlg = decoder.read(new ByteArrayInputStream(data)) == 0;
            AtomicReference<class_2960> tex = new AtomicReference<>();
            if (!gifFlg) {
                class_1011 ni = class_1011.method_4309(new ByteArrayInputStream(data));
                mc.method_20493(() -> tex.set(mc.method_1531().method_4617("native_texture", new class_1043(ni)))).get();
            } else {
                DynamicGifTexture.ImageFrame[] frames = new DynamicGifTexture.ImageFrame[decoder.getFrameCount()];
                long duration = 0;
                for (int i = 0; i < decoder.getFrameCount(); i++) {
                    frames[i] = new DynamicGifTexture.ImageFrame(class_1011.method_4309(FNImageUtil.toInputStream(decoder.getFrame(i), "png")), decoder.getDelay(i));
                    duration += decoder.getDelay(i);
                }
                long finalDuration = duration;
                mc.method_20493(() -> tex.set(mc.method_1531().method_4617("native_texture", new DynamicGifTexture(finalDuration, frames)))).get();
            }
            stream.close();
            return new TextureLoadResult(tex.get(), false);
        } catch (Exception ex) {
            ex.printStackTrace();
        }
        return new TextureLoadResult(failureTexture, true);
    }

    /**
     * ネイティブテクスチャを開放する
     *
     * @param id テクスチャ管理ID
     */
    public static void freeNativeTexture(UUID id) {
        if (NATIVE_TEXTURES.containsKey(id)) {
            if (!NATIVE_TEXTURES.get(id).failure())
                freeTexture(NATIVE_TEXTURES.get(id).location());
            NATIVE_TEXTURES.remove(id);
        }
    }

    /**
     * テクスチャを開放する
     *
     * @param location テクスチャID
     */
    public static void freeTexture(class_2960 location) {
        OEClientExpectPlatform.freeTexture(location);
    }

    private static record TextureLoadResult(class_2960 location, boolean failure) {
    }

    public static record TextureScale(double w, double h) {
    }
}
