package dev.felnull.otyacraftengine.util;

import ;
import dev.felnull.otyacraftengine.server.level.TagSerializable;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.Function;
import java.util.function.Supplier;
import net.minecraft.class_2487;
import net.minecraft.class_2499;
import net.minecraft.class_2512;
import net.minecraft.class_2520;
import net.minecraft.class_3542;

/**
 * NBT関係のユーティリティ
 *
 * @author MORIMORI0317
 */
public class OENbtUtils {
    /**
     * UUIDのTAG番号
     */
    public static final int TAG_UUID = class_2520.field_33261;

    /**
     * NBTにリストを書き込む
     *
     * @param tag    NBTタグ
     * @param name   リストの名前
     * @param list   リスト
     * @param writer 書き込み方法
     * @param <T>    　リストの種類
     * @return 書き込み済みNBT
     */
    @NotNull
    public static <T> class_2487 writeList(@NotNull class_2487 tag, @NotNull String name, @Nullable List<T> list, @NotNull Function<T, class_2520> writer) {
        if (list != null) {
            class_2499 listTag = new class_2499();
            list.forEach(n -> listTag.add(writer.apply(n)));
            tag.method_10566(name, listTag);
        }
        return tag;
    }

    /**
     * NBTにUUIDのリストを書き込む
     *
     * @param tag   NBTタグ
     * @param name  リストの名前
     * @param uuids UUIDリスト
     * @return 書き込み済みNBT
     */
    @NotNull
    public static class_2487 writeUUIDList(@NotNull class_2487 tag, @NotNull String name, @Nullable List<UUID> uuids) {
        return writeList(tag, name, uuids, class_2512::method_25929);
    }

    /**
     * NBTからリストを読み込む
     *
     * @param tag    NBTタグ
     * @param name   リストの名前
     * @param list   書き込むリスト
     * @param reader 読み込み方法
     * @param num    TAG番号
     * @param <T>    リストの種類
     * @return 書き込み済みリスト
     */
    @NotNull
    public static <T> List<T> readList(@NotNull class_2487 tag, @NotNull String name, @Nullable List<T> list, @NotNull Function<class_2520, T> reader, int num) {
        if (list != null) list.clear();
        else list = new ArrayList<>();

        if (tag.method_10573(name, class_2520.field_33259)) {
            class_2499 listTag = tag.method_10554(name, num);
            for (class_2520 lstag : listTag) {
                list.add(reader.apply(lstag));
            }
        }

        return list;
    }

    /**
     * NBTからUUIDリストを読み込む
     *
     * @param tag   NBTタグ
     * @param name  リストの名前
     * @param uuids UUIDリスト
     * @return 書き込み済みリスト
     */
    @NotNull
    public static List<UUID> readUUIDList(@NotNull class_2487 tag, @NotNull String name, @Nullable List<UUID> uuids) {
        return readList(tag, name, uuids, class_2512::method_25930, TAG_UUID);
    }

    /**
     * NBTにTagSerializableを実装した値のリストを書き込む
     *
     * @param tag              NBTタグ
     * @param name             リストの名前
     * @param serializableList TagSerializableを実装した値リスト
     * @param <T>              TagSerializableを実装した値
     * @return 書き込み済みNBT
     */
    @NotNull
    public static <T extends TagSerializable> class_2487 writeSerializableList(@NotNull class_2487 tag, @NotNull String name, @Nullable List<T> serializableList) {
        return writeList(tag, name, serializableList, TagSerializable::createSavedTag);
    }

    /**
     * NBTからTagSerializableを実装した値のリストを読み込む
     *
     * @param tag              NBTタグ
     * @param name             リストの名前
     * @param serializableList TagSerializableを実装した値リスト
     * @param initSerializable TagSerializableを実装した値の初期
     * @param <T>              TagSerializableを実装した値
     * @return 書き込み済みリスト
     */
    @NotNull
    public static <T extends TagSerializable> List<T> readSerializableList(@NotNull class_2487 tag, @NotNull String name, @Nullable List<T> serializableList, @NotNull Supplier<T> initSerializable) {
        return readList(tag, name, serializableList, tag1 -> TagSerializable.loadSavedTag((class_2487) tag1, initSerializable.get()), class_2520.field_33260);
    }

    /**
     * NBTにマップを書き込む
     *
     * @param tag         NBTタグ
     * @param name        マップの名前
     * @param map         マップ
     * @param keyWriter   キーの書き込み方法
     * @param valueWriter 　値の書き込み方法
     * @param <T>         キー
     * @param <M>         値
     * @return 書き込み済みNBT
     */
    @NotNull
    public static <T, M> class_2487 writeMap(@NotNull class_2487 tag, @NotNull String name, @Nullable Map<T, M> map, @NotNull Function<T, class_2520> keyWriter, @NotNull Function<M, class_2520> valueWriter) {
        if (map != null) {
            List<T> key = new ArrayList<>(map.keySet());
            List<M> value = new ArrayList<>(map.values());
            var mt = new class_2487();

            writeList(mt, "k", key, keyWriter);
            writeList(mt, "v", value, valueWriter);
            tag.method_10566(name, mt);
        }
        return tag;
    }

    /**
     * NBTからマップを読み込む
     *
     * @param tag         NBTタグ
     * @param name        マップの名前
     * @param map         マップ
     * @param keyReader   キーの読み込み方法
     * @param valueReader 値の読み込み方法
     * @param keyNum      キーのTAG番号
     * @param valueNum    値のTAG番号
     * @param <T>         キー
     * @param <M>         値
     * @return 書き込み済みマップ
     */
    @NotNull
    public static <T, M> Map<T, M> readMap(@NotNull class_2487 tag, @NotNull String name, @Nullable Map<T, M> map, @NotNull Function<class_2520, T> keyReader, @NotNull Function<class_2520, M> valueReader, int keyNum, int valueNum) {
        if (map != null) map.clear();
        else map = new HashMap<>();
        var mt = tag.method_10562(name);
        var kr = readList(mt, "k", null, keyReader, keyNum);
        var vr = readList(mt, "v", null, valueReader, valueNum);
        if (kr.size() != vr.size()) throw new IllegalArgumentException("The count of key and value do not match.");

        for (int i = 0; i < kr.size(); i++) {
            var k = kr.get(i);
            var v = vr.get(i);
            map.put(k, v);
        }
        return map;
    }

    /**
     * NBTにUUIDがキーのマップを書き込む
     *
     * @param tag         NBTタグ
     * @param name        マップの名前
     * @param map         マップ
     * @param valueWriter 値の書き込み方法
     * @param <T>         値
     * @return 書き込み済みNBT
     */
    @NotNull
    public static <T> class_2487 writeUUIDKeyMap(@NotNull class_2487 tag, @NotNull String name, @Nullable Map<UUID, T> map, @NotNull Function<T, class_2520> valueWriter) {
        return writeMap(tag, name, map, class_2512::method_25929, valueWriter);
    }

    /**
     * NBTからUUIDがキーのマップを読み込む
     *
     * @param tag         NBTタグ
     * @param name        マップの名前
     * @param map         マップ
     * @param valueReader 値の読み込み方法
     * @param valueNum    値のTAG番号
     * @param <T>         値
     * @return 読み込み済みマップ
     */
    @NotNull
    public static <T> Map<UUID, T> readUUIDKeyMap(@NotNull class_2487 tag, @NotNull String name, @Nullable Map<UUID, T> map, @NotNull Function<class_2520, T> valueReader, int valueNum) {
        return readMap(tag, name, map, class_2512::method_25930, valueReader, TAG_UUID, valueNum);
    }

    /**
     * NBTにUUIDマップを書き込む
     *
     * @param tag  NBTタグ
     * @param name マップの名前
     * @param map  マップ
     * @return 書き込み済みNBT
     */
    @NotNull
    public static class_2487 writeUUIDMap(@NotNull class_2487 tag, @NotNull String name, @Nullable Map<UUID, UUID> map) {
        return writeUUIDKeyMap(tag, name, map, class_2512::method_25929);
    }

    /**
     * NBTからUUIDマップを読み込む
     *
     * @param tag  NBTタグ
     * @param name マップの名前
     * @param map  マップ
     * @return 読み込み済みマップ
     */
    @NotNull
    public static Map<UUID, UUID> readUUIDMap(@NotNull class_2487 tag, @NotNull String name, @Nullable Map<UUID, UUID> map) {
        return readUUIDKeyMap(tag, name, map, class_2512::method_25930, TAG_UUID);
    }


    @NotNull
    public static class_2487 writeUUIDTagMap(@NotNull class_2487 tag, @NotNull String name, @Nullable Map<UUID, class_2487> map) {
        return writeUUIDKeyMap(tag, name, map, tag1 -> tag1);
    }

    @NotNull
    public static Map<UUID, class_2487> readUUIDTagMap(@NotNull class_2487 tag, @NotNull String name, @Nullable Map<UUID, class_2487> map) {
        return readUUIDKeyMap(tag, name, map, tag1 -> (class_2487) tag1, class_2520.field_33260);
    }

    /**
     * NBTにUUIDがキーでTagSerializableを実装した値のマップを書き込む
     *
     * @param tag  NBTタグ
     * @param name マップの名前
     * @param map  マップ
     * @param <T>  TagSerializableを実装した値
     * @return 書き込み済みNBT
     */
    @NotNull
    public static <T extends TagSerializable> class_2487 writeUUIDSerializableMap(@NotNull class_2487 tag, @NotNull String name, @Nullable Map<UUID, T> map) {
        return writeUUIDKeyMap(tag, name, map, TagSerializable::createSavedTag);
    }

    /**
     * NBTからUUIDがキーでTagSerializableを実装した値のマップを読み込む
     *
     * @param tag              NBTタグ
     * @param name             マップの名前
     * @param map              マップ
     * @param initSerializable TagSerializableを実装した値の初期
     * @param <T>              TagSerializableを実装した値
     * @return 読み込み済みマップ
     */
    @NotNull
    public static <T extends TagSerializable> Map<UUID, T> readUUIDSerializableMap(@NotNull class_2487 tag, @NotNull String name, @Nullable Map<UUID, T> map, @NotNull Supplier<T> initSerializable) {
        return readUUIDKeyMap(tag, name, map, tag1 -> TagSerializable.loadSavedTag((class_2487) tag1, initSerializable.get()), class_2520.field_33260);
    }

    /**
     * 順序数で列挙型をNBTタグに書き込む
     * MODのバージョン違いなどの影響で列挙型の数や配置が違う場合に別の値になってしまう可能性があります
     * 通信など列挙型の数や配置が完全に一致することを保障できる場合に利用を推奨
     * ワールドデータの保存などMODバージョンが違って読み込む可能性がある場合は {@link #writeEnum(class_2487, String, Enum)} を利用してください
     *
     * @param tag   NBTタグ
     * @param name  保存名
     * @param enum_ 列挙型
     * @return 書き込み済みNBT
     */
    @NotNull
    public static class_2487 writeEnumByOrdinal(@NotNull class_2487 tag, @NotNull String name, @NotNull Enum<?> enum_) {
        tag.method_10569(name, enum_.ordinal());
        return tag;
    }

    /**
     * 順序数で列挙型をNBTタグから読み込む
     * MODのバージョン違いなどの影響で列挙型の数や配置が違う場合に別の値になってしまう可能性があります
     * 通信など列挙型の数や配置が完全に一致することを保障できる場合に利用を推奨
     * ワールドデータの保存などMODバージョンが違って読み込む可能性がある場合は
     *
     * @param tag         NBTタグ
     * @param name        保存名
     * @param enumClass   列挙型クラス
     * @param defaultEnum デフォルト列挙型
     * @param <T>         列挙型
     * @return 読み込まれた列挙型
     */
    public static <T extends Enum<T>> T readEnumByOrdinal(@NotNull class_2487 tag, @NotNull String name, @NotNull Class<T> enumClass, @Nullable T defaultEnum) {
        var ens = ((T[]) enumClass.getEnumConstants());
        int num = tag.method_10550(name);
        if (num < 0 || ens.length <= num) return defaultEnum;
        return ens[num];
    }

    /**
     * シリアル名で列挙型をNBTタグに書き込む
     *
     * @param tag   NBTタグ
     * @param name  保存名
     * @param enum_ 列挙型
     * @return 書き込み済みNBT
     */
    @NotNull
    public static class_2487 writeEnum(@NotNull class_2487 tag, @NotNull String name, @NotNull Enum<? extends class_3542> enum_) {
        tag.method_10582(name, ((class_3542) (enum_)).method_15434());
        return tag;
    }

    /**
     * シリアル名で列挙型をNBTタグから読み込む
     *
     * @param tag         NBTタグ
     * @param name        保存名
     * @param enumClass   列挙型クラス
     * @param defaultEnum デフォルト列挙型
     * @param <T>         列挙型
     * @return 読み込まれた列挙型
     */
    public static <T extends Enum<T> & class_3542> T readEnum(@NotNull class_2487 tag, @NotNull String name, @NotNull Class<T> enumClass, @Nullable T defaultEnum) {
        var ens = ((T[]) enumClass.getEnumConstants());
        var n = tag.method_10558(name);
        for (T en : ens) {
            if (en.method_15434().equals(n))
                return en;
        }
        return defaultEnum;
    }


}
